<?php







declare(strict_types=1);







namespace Apaczka\Api\Client\Http;







use Apaczka\Api\Client\Http\Exception\HttpBadRequestException;



use Apaczka\Api\Client\Http\Plugin\PluginInterface;



use Apaczka\Api\Client\Http\Processor\RequestProcessorProviderInterface;



use GuzzleHttp\Exception\GuzzleException;



use Psr\Http\Message\RequestFactoryInterface;



use Psr\Http\Message\RequestInterface;



use Psr\Http\Message\ResponseInterface;



use Psr\Http\Message\StreamFactoryInterface;



use Psr\Http\Message\StreamInterface;



use Psr\Http\Message\UriInterface;



use stdClass;







class HttpClient



{



	private RequestFactoryInterface $requestFactory;



	private StreamFactoryInterface $streamFactory;



	private RequestProcessorProviderInterface $requestProcessorProvider;







	/**



	 * @var PluginInterface[]



	 */



	private array $plugins = [];







	public function __construct(



		RequestProcessorProviderInterface $requestProcessorProvider,



		RequestFactoryInterface $requestFactory,



		StreamFactoryInterface $streamFactory,



		PluginInterface ...$plugins



	) {



		$this->requestFactory = $requestFactory;



		$this->streamFactory = $streamFactory;



		$this->plugins = $plugins;



		$this->requestProcessorProvider = $requestProcessorProvider;



	}







	public function get(UriInterface $uri): ResponseInterface



	{



		return $this->process(



			$this->createRequest(new HttpMethodEnum(HttpMethodEnum::GET), $uri)



		);



	}







	/**



	 * @param array<string, string|int|float|bool|stdClass|null> $data



	 */



	public function post(UriInterface $uri, array $data = []): ResponseInterface



	{



		return $this->process(



			$this->createRequest(



                new HttpMethodEnum(HttpMethodEnum::POST),



                $uri,



                $data



            )



			->withHeader('Content-Type', 'application/x-www-form-urlencoded')



		);



	}







	/**



	 * @param array<string, string|int|float|bool|stdClass|null> $body



	 */



	private function createRequest(HttpMethodEnum $method, UriInterface $uri, array $body = []): RequestInterface



	{



		$request = $this->requestFactory->createRequest((string) $method, (string) $uri);







		return $this->applyPluginsToRequest(



			$this->withBody($request, $body)



		);



	}







	private function process(RequestInterface $request, int $successCode = 200): ResponseInterface



	{



        $request->getBody()->rewind();







		$response = $this->requestProcessorProvider->getProcessor()->process($request, $successCode);







		$response = $this->applyPluginsToResponse(



			$request,



			$response



		);







		$body = $response->getBody()->getContents();



		$response->getBody()->rewind();



		$data = json_decode($body, false);







		if (is_object($data) && isset($data->status) && $data->status !== $successCode) {



			throw new HttpBadRequestException(



				isset($data->message) ? (string) $data->message : 'Unknown error',



				(int) $data->status



			);



		}







		return $response;



	}







	/**



	 * @param array<string, string|int|float|bool|null> $body



	 */



	private function withBody(RequestInterface $request, array $body = []): RequestInterface



	{



		if (0 === count($body)) {



			return $request;



		}







		return $request->withBody($this->prepareBody($body));



	}







	/**



	 * @param array<string, string|int|float|bool|null> $body



	 */



	private function prepareBody(array $body): StreamInterface



	{



		return $this->streamFactory->createStream(



			json_encode($body)



		);



	}







	private function applyPluginsToRequest(RequestInterface $request): RequestInterface



	{



		foreach ($this->plugins as $plugin) {



			$request = $plugin->handleRequest($request);



		}







		return $request;



	}







	private function applyPluginsToResponse(RequestInterface $request, ResponseInterface $response): ResponseInterface



	{



		foreach ($this->plugins as $plugin) {



			$response = $plugin->handleResponse($request, $response);



		}







		return $response;



	}



}



