<?php







declare(strict_types=1);







namespace Apaczka\Api\Client;







use Apaczka\Api\Client\Helper\DataTransformerHelper;



use Apaczka\Api\Client\Http\HttpClient;



use Apaczka\Api\Client\Model\Order;



use Apaczka\Api\Client\Model\OrderCollection;



use Apaczka\Api\Client\Model\PointsCollection;



use Apaczka\Api\Client\Model\PointTypeEnum;



use Apaczka\Api\Client\Model\PriceCollection;



use Apaczka\Api\Client\Model\ServiceStructure;



use Apaczka\Api\Client\Model\TurnIn;



use Apaczka\Api\Client\Model\Waybill;



use Apaczka\Api\Client\Settings\SettingsInterface;



use Apaczka\Api\Client\Transformer\CollectionTransformer;



use Apaczka\Api\Client\Transformer\IdsCollectionTransformer;



use Apaczka\Api\Client\Transformer\OrderTransformer;



use Apaczka\Api\Client\Transformer\PointTransformer;



use Apaczka\Api\Client\Transformer\PriceTransformer;



use Apaczka\Api\Client\Transformer\ServiceStructureTransformer;



use Apaczka\Api\Client\Transformer\TransformerInterface;



use Apaczka\Api\Client\Transformer\TurnInTransformer;



use Apaczka\Api\Client\Transformer\WaybillTransformer;



use Psr\Http\Message\ResponseInterface;



use Psr\Http\Message\UriInterface;



use stdClass;







class ApiClient



{



	public const MAX_LIMIT = 25;







	private SettingsInterface $settings;



	private HttpClient $httpClient;







    private array $cached= [];







	public function __construct(SettingsInterface $settings, HttpClient $httpClient)



	{



		$this->httpClient = $httpClient;



		$this->settings = $settings;



	}







	public function getServiceStructure(): ServiceStructure



	{



        $path = 'service_structure/';



        if (!isset($this->cached[$path])) {



            $this->cached[$path] = $this->processResponse(



                $this->httpClient->post($this->createUrl($path)),



                new ServiceStructureTransformer()



            );



        }







        return $this->cached[$path];



	}







	public function getPoints(PointTypeEnum $type): PointsCollection



	{



		return $this->processResponse(



			$this->httpClient->post($this->createUrl(sprintf('points/%s/', $type))),



			new IdsCollectionTransformer(



				new PointTransformer(),



				new PointsCollection()



			),



			['points']



		);



	}







	public function getOrders(int $page = 1, int $limit = self::MAX_LIMIT): OrderCollection



	{



		return $this->processResponse(



			$this->httpClient->post($this->createUrl('orders/'), [



				'page' => $page,



				'limit' => $limit



			]),



			new CollectionTransformer(



				new OrderTransformer(),



				new OrderCollection()



			),



			['orders']



		);



	}







	public function getValuation(Order $order): PriceCollection



	{



		$transformer = new OrderTransformer();



		$data = DataTransformerHelper::remapKeys(



            $transformer->revert($order),



            ['shipments' => 'shipment']



        );







		$response =  $this->processResponse(



			$this->httpClient->post(



                $this->createUrl('order_valuation/'),



                [



				    'order' => $data,



			    ]



            ),



			new IdsCollectionTransformer(



				new PriceTransformer(),



				new PriceCollection()



			),



			['price_table']



		);







        return $response;



	}







	public function send(Order $order): Order



	{



		$transformer = new OrderTransformer();







		$data = DataTransformerHelper::remapKeys($transformer->revert($order), [



			'shipments' => 'shipment',



		]);







		return $this->processResponse(



			$this->httpClient->post($this->createUrl('order_send/'), [



				'order' => $data,

				'system' => 'prestashop17'

			]),



			new OrderTransformer(),



			['order']



		);



	}







	public function getOrder(string $orderNumber): Order



	{



		return $this->processResponse(



			$this->httpClient->post($this->createUrl(sprintf('order/%s/', $orderNumber))),



			new OrderTransformer(),



			['order']



		);



	}







    public function cancelOrder(string $orderNumber): void



    {



        $this->processResponse(



            $this->httpClient->post($this->createUrl(sprintf('cancel_order/%s/', $orderNumber))),



            null,



            []



        );



    }







    public function getWaybill(string $orderNumber): ?Waybill



    {



        return $this->processResponse(



            $this->httpClient->post($this->createUrl(sprintf('waybill/%s/', $orderNumber))),



            new WaybillTransformer(),



            []



        );



    }







    public function getTurnIn(array $orderNumbers): ?TurnIn



    {



        return $this->processResponse(



            $this->httpClient->post($this->createUrl('turn_in'), ['order_ids' => $orderNumbers]),



            new TurnInTransformer(),



            []



        );



    }







	/**



	 * @param TransformerInterface<mixed,mixed> $transformer



	 * @param string[] $propertyPath



	 * @return mixed



	 */



	private function processResponse(ResponseInterface $response, ?TransformerInterface $transformer = null, array $propertyPath = [])



	{



		$payload = json_decode($response->getBody()->getContents());



		$data = $payload->response;







		foreach ($propertyPath as $property) {



			$data = is_object($data) && property_exists($data, $property)



				? $data->$property // @phpstan-ignore-line



				: null;



		}







		if (null === $transformer) {



			return $data;



		}







		return $transformer->transform($data);



	}







	private function createUrl(string $path): UriInterface



	{



		$separator = '/';







		return $this->settings->getApiUrl()->withPath(



			implode(



				$separator,



				[



				rtrim($this->settings->getApiUrl()->getPath(), $separator),



				ltrim($path, $separator)]



			)



		);



	}



}